// main.cpp
// A minimalist C++/WTL web browser using the WebView2 engine.

#define UNICODE
#define _UNICODE

// Target Windows 10 and later.
#define _WIN32_WINNT 0x0A00
#include <windows.h>
#include <atlbase.h>
#include <atlcom.h>

// --- FIX 1: INCLUDE ATLTYPES.H FOR CSIZE ---
// This header file defines the CSize and CPoint classes used by WTL message maps.
#include <atltypes.h>

#include <atlapp.h>
#include <atlwin.h>
#include <atlcrack.h>
#include <atlframe.h>
#include <atlctrls.h>
#include <wrl.h> 
#include <wrl/client.h>
#include <wil/com.h>

// Include the WebView2 header.
#include "WebView2.h"

// Global Module object for WTL
CAppModule _Module;

#define ID_GO 1

// Main Window Class
class CMainWindow : public CFrameWindowImpl<CMainWindow>
{
public:
    // WTL window class declaration
    DECLARE_WND_CLASS(L"MinimalBrowser")

    // WebView2 specific members
    wil::com_ptr<ICoreWebView2Controller> m_controller;
    wil::com_ptr<ICoreWebView2> m_webview;

    // UI Elements
    CEdit m_addressBar;

    // WTL Message Map
    BEGIN_MSG_MAP(CMainWindow)
        MSG_WM_CREATE(OnCreate)
        MSG_WM_SIZE(OnSize)
        COMMAND_ID_HANDLER(ID_GO, OnGo) 
        CHAIN_MSG_MAP(CFrameWindowImpl<CMainWindow>)
    END_MSG_MAP()

    // Window Creation Handler
    LRESULT OnCreate(LPCREATESTRUCT lpCreateStruct)
    {
        m_hWndClient = m_addressBar.Create(m_hWnd, rcDefault, L"https://www.google.com", 
            WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | ES_AUTOHSCROLL, 0, 12345);

        SetMsgHandled(FALSE);

        CreateCoreWebView2EnvironmentWithOptions(nullptr, nullptr, nullptr,
            Microsoft::WRL::Callback<ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler>(
                [this](HRESULT result, ICoreWebView2Environment* env) -> HRESULT {
                    
                    env->CreateCoreWebView2Controller(m_hWnd, Microsoft::WRL::Callback<ICoreWebView2CreateCoreWebView2ControllerCompletedHandler>(
                        [this](HRESULT result, ICoreWebView2Controller* controller) -> HRESULT {
                            if (controller != nullptr) {
                                m_controller = controller;
                                m_controller->get_CoreWebView2(&m_webview);
                            }

                            RECT bounds;
                            GetClientRect(&bounds);
                            m_controller->put_Bounds(bounds);
                            
                            OnSize(0, CSize(bounds.right, bounds.bottom));

                            m_webview->Navigate(L"https://www.google.com");

                            return S_OK;
                        }).Get());
                    return S_OK;
                }).Get());

        return 0;
    }

    // Window Resize Handler
    void OnSize(UINT nType, CSize size)
    {
        if (m_controller == nullptr) 
		{
			SetMsgHandled(FALSE); // Must call this even if we do nothing
			return;
		}

        const int addressBarHeight = 24;
        ::SetWindowPos(m_addressBar, NULL, 0, 0, size.cx, addressBarHeight, SWP_NOZORDER);

        RECT wvBounds = { 0, addressBarHeight, size.cx, size.cy };
        m_controller->put_Bounds(wvBounds);
		
		SetMsgHandled(FALSE);
    }
    
    // Handler for navigation
    LRESULT OnGo(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
    {
        if (m_webview == nullptr) return 0;
        
        wchar_t url[2048];
        m_addressBar.GetWindowText(url, 2048);

        m_webview->Navigate(url);
        
        return 0;
    }
};

// Main application entry point
int WINAPI wWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PWSTR pCmdLine, int nCmdShow)
{
    CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);
    _Module.Init(NULL, hInstance);

    CMessageLoop msgLoop;
    _Module.AddMessageLoop(&msgLoop);

    CMainWindow win;
	// --- FIX 2: CORRECTED TYPO ---
    if (win.Create(NULL, CWindow::rcDefault, L"Minimal Browser", WS_OVERLAPPEDWINDOW | WS_VISIBLE) == NULL)
    {
        ATLTRACE(L"Main window creation failed!\n");
        return 0;
    }
	
	win.ShowWindow(nCmdShow);
    win.UpdateWindow();

    int nRet = msgLoop.Run();

    _Module.RemoveMessageLoop();
    _Module.Term();
    CoUninitialize();

    return nRet;
}